//
//  PaymentResult.h
//  PayKKaCheckoutPayments
//
//  Created by jiangzheng on 2025/11/27.
//

#import <Foundation/Foundation.h>

NS_ASSUME_NONNULL_BEGIN

/// 支付状态枚举
typedef NS_ENUM(NSInteger, PaymentStatus) {
    /// 未知状态
    PaymentStatusUnknown = -1,
    /// 支付成功
    PaymentStatusSuccess = 0,
    /// 支付已过期
    PaymentStatusExpired = 1,
    /// 支付发生错误或失败
    PaymentStatusError = 2
};

/// 支付结果信息
@interface PaymentResult : NSObject

#pragma mark - 属性

/// 支付状态
@property (nonatomic, assign, readonly) PaymentStatus status;

/// 原始支付结果数据
@property (nonatomic, strong, readonly, nullable) NSDictionary *result;

/// 支付提示或错误信息
@property (nonatomic, copy, readonly, nullable) NSString *message;

/// 包含状态、结果和信息的完整支付结果字典
@property (nonatomic, strong, readonly) NSDictionary *paymentResultObj;

#pragma mark - 初始化

/// 使用支付状态、原始结果和提示信息初始化支付结果实例
///
/// - Parameters:
///   - status: 支付状态，参考 `PaymentStatus`
///   - result: 支付返回的原始字典数据
///   - message: 支付提示信息或错误描述
/// - Returns: `PaymentResult` 实例
- (instancetype)initWithStatus:(PaymentStatus)status
                        result:(NSDictionary * _Nullable)result
                       message:(NSString * _Nullable)message;

/// 生成标准化的支付结果字典
///
/// - Parameters:
///   - status: 支付状态
///   - data: 支付返回的原始字典数据
///   - altData: 备用数据列表
///   - message: 支付提示信息
/// - Returns: 标准化的 `NSDictionary`，包含 `status`、`result` 和 `message` 字段
+ (NSDictionary *)makeResult:(PaymentStatus)status
                        data:(NSDictionary * _Nullable)data
                     altData:(NSArray * _Nullable)altData
                     message:(NSString * _Nullable)message;

#pragma mark - JSON 解析与序列化

/// 从 JSON 字符串解析并创建支付结果实例
///
/// - Parameter jsonString: 符合格式要求的 JSON 字符串
/// - Returns: `PaymentResult` 实例
+ (instancetype)fromString:(NSString *)jsonString;

/// 将当前支付结果对象序列化为 JSON 字符串
///
/// - Returns: 包含 `status`、`result` 和 `message` 的 JSON 格式字符串
- (NSString *)toString;

@end

NS_ASSUME_NONNULL_END
